#include "GUI/SimpleTest.h"
#include "Grabbag/GrabbagTester.h"
#include "../RecursiveFunctions.h"
#include <vector>
#include <sstream>
#include <algorithm>
#include <iterator>
using namespace std;

namespace {
    /* Rabin rolling hash code. Our implementation works by checking your answers against
     * a series of hash codes for the correct answer. This gives high confidence that your
     * answer is correct without actually revealing what the answer is. :-)
     */
    uint32_t rollingHash(const string& str, uint64_t base, uint64_t multiplier) {
        const uint64_t modulus = 0x7FFFFFFF;
        uint64_t result = base;
        for (char ch: str) {
            result = (((result * multiplier) % modulus) + uint8_t(ch)) % modulus;
        }
        return result % modulus;
    }

    /* Given a list of values, produces a key for those answers. */
    string keyFor(const vector<int>& values) {
        ostringstream builder;
        copy(values.begin(), values.end(), ostream_iterator<int>(builder, ","));
        return builder.str();
    }
}

PROVIDED_TEST("Check compute_h on simple cases.") {
    EXPECT_EQUAL(compute_h(0), 0);
    EXPECT_EQUAL(compute_h(1), 1);
    EXPECT_EQUAL(compute_h(2), 3);
    EXPECT_EQUAL(compute_h(3), 6);
    EXPECT_EQUAL(compute_h(4), 7);
    EXPECT_EQUAL(compute_h(5), 12);
    EXPECT_EQUAL(compute_h(6), 14);
    EXPECT_EQUAL(compute_h(7), 15);
    EXPECT_EQUAL(compute_h(8), 21);
    EXPECT_EQUAL(compute_h(9), 24);
    EXPECT_EQUAL(compute_h(10), 28);
}

PROVIDED_TEST("Check compute_h over a wider range of inputs.") {
    /* Sample the values of h at various inputs. */
    vector<int> values;
    for (int i = 0; i < 137103; i += 271) {
        values.push_back(compute_h(i));
    }

    auto key = keyFor(values);

    /* Confirm this matches what we want it to be. */
    runPrivateTest("compute_h", [&](istream& input) {
        for (uint64_t base, multiplier, result; input >> base >> multiplier >> result; ) {
            /* Hash must agree; if not, we are definitely wrong. */
            if (rollingHash(key, base, multiplier) != result) {
                SHOW_ERROR("Answer is incorrect.");
            }
        }
    });
}
